##  Author: Stefan Evert
## Purpose: Evaluate ranking of MWE candidates by plotting precision against recall

## line styles for colour plots
.style.col <-
  list(col = rep(c("#000000", "#FF0000", "#0000FF", "#008800", "#FF6600"), 2),
       lty = c(rep("solid", 5), rep("dashed", 5)),
       lwd = rep(2, 10))

## line styles for b/w plots
.style.bw <-
  list(col = c(rep("black", 4), rep("#666666", 4), "black", "#666666"),
       lty = c("solid","33","11","dotdash","solid","33","11","dotdash", "longdash","longdash"),
       lwd = c(2,3,4,3, 3,3,4,3, 3,3))

precision.recall.plot <- function (TP, Scores, legend=NULL, ylim=c(0,100), bw=FALSE, min.recall=5,
                                   title="Precision-recall graph") {
  if (missing(legend) && is.list(Scores) && !is.null(names(Scores))) {
    legend <- names(Scores)
  }
  if (! is.list(Scores)) {
    stopifnot(is.numeric(Scores))
    Scores <- list(Scores)
  }
  TP <- as.logical(TP)
  total.tp <- sum(TP)
  n.cand <- length(TP)
  baseline <- 100 * total.tp / n.cand   # baseline precision
  
  style <- if (bw) .style.bw else .style.col
  
  n.rankings <- length(Scores)
  if (! missing(legend)) {
    stopifnot(length(legend) == n.rankings)
    stopifnot(n.rankings <= length(style$lty))
  }
  
  plot(0,0, type="n", xlim=c(0,100), ylim=ylim, xaxs="i", yaxs="i",
       main=title, xlab="Recall (%)", ylab="Precision (%)")
  abline(h=baseline, lwd=2, lty="dotted")
  
  num <- 1
  avg.prec <- c()
  for (s in Scores) {
    stopifnot(length(s) == n.cand)
    ranking <- order(s, decreasing=TRUE)
    s.sorted <- s[ranking]
    idx <- c(diff(s.sorted) != 0, TRUE) # n-best sets that correspond to cutoff thresholds
    n.tp <- cumsum(TP[ranking])
    precision <- n.tp / (1:n.cand)
    recall <- n.tp / total.tp
    x.coords <- 100 * recall[idx]       # coordinates of precision/recall points to be plotted
    y.coords <- 100 * precision[idx]
    to.show <- x.coords >= min.recall
    lines(x.coords[to.show], y.coords[to.show], type="S", lty=style$lty[num], lwd=style$lwd[num], col=style$col[num])
    avg.prec[num] <- sum(diff(c(0, x.coords)) * y.coords) / 100 # average precision (%) = area under precision-recall curve
    num <- num + 1
  }

  if (! is.null(legend)) {
    names(avg.prec) <- legend
    legend("bottomleft", inset=.05, legend=legend, bg="white",
           lty=style$lty[1:n.rankings], lwd=style$lwd[1:n.rankings], col=style$col[1:n.rankings])
  }
  cat("Average precision (%):\n")
  print(round(avg.prec, 2))
  cat("(baseline: ", round(baseline, 2), "%)\n", sep="")
}
