##  Author: Stefan Evert
## Purpose: Demonstrate usage of evaluation functions in "precision_recall.R"

## load support functions for evaluation graphs
source("precision_recall.R")

## read gold standard file with one or more automatic rankings 
##  - must be a TAB-delimited table with single header row (column names are not important)
##  - rankings given in the form of (association) scores, with high scores indicating good candidates
AN <- read.delim("de_an.rankings.tbl")

## open graphics window and set display parameters (you may want to use another driver, e.g. quartz(...) on Mac OS X)
x11(width=8, height=8)  # you can also choose a rectangular window (non-square)
par(cex=1.3)            # increase font size: use 1.3 for figures in conference paper, 1.5 for very small figures
par(mar=c(4,4,3,1)+.1)  # plot margins: you probably don't want to change these

## vector true positives = categories 1-3 in manual annotation (column n.cat)
TP <- AN$n.cat <= 3

## vector of association scores
chisq <- AN$am.chi.squared.corr         # chi-squared (with Yates' correction) in column am.chi.squared.corr
frequency <- AN$am.frequency            # ranking by observed frequency (has many ties!) in column am.frequency

## plot precision-recall graph (automatically displays average precision in command window!)
precision.recall.plot(TP, chisq)

## combine multiple rankings in list to show all graphs in single plot
precision.recall.plot(TP, list(chisq, frequency))

## label rankings to display legend box
precision.recall.plot(TP, list(X2=chisq, f=frequency))

## can also pass table ("data.frame") with all desired rankings
scores <- AN[,4:8]                      # association scores are in columns 4 .. 8 of the table
precision.recall.plot(TP, scores)

## in this case, specify explicit "legend" option to get nicer names
names <- c("G2", "X2", "MI", "Dice", "freq") # simple text legend
names <- expression(G^2, X^2, "MI", "Dice", f) # alternatively, use mathematical formatting
precision.recall.plot(TP, scores, names)

## small range of "interesting" precision values => "zoom in" on y-axis
precision.recall.plot(TP, scores, names, ylim=c(30,80))

## switch to b/w mode and add descriptive main title for print version
precision.recall.plot(TP, scores, names, ylim=c(30,80), bw=TRUE, title="German AN (cat. 1-3), standard association measures")

## on most platforms, you can copy & paste the plot or save it to a file (.pdf, .wmf, etc.)

## you can also use the following R commands to generate a PDF version of the on-screen plot
##  - option onefile=FALSE is important to get a useful PDF image for inclusion into other documents
##  - width and height must match the values specified when opening the graphics device
##  - add option bg="white" if you want to use image in screen presentation with dark background
dev.copy(pdf, file="example_figure.pdf", onefile=FALSE, width=8, height=8)
dev.off()

## check comments in "precision_recall.R" if you want to customise line styles or set other options

