#!/usr/bin/perl

###############################################################################
#
#                        The MWE 2008 Shared Task
#
# This is a "QuickStart" script for evaluating a ranked list of MWE candidates
# extracted from the Prague Dependency Treebank 2.0. It computes Average
# Precision for the full n-best list based on the Ground Truth data.
#
# Examples usage: ./eval.pl rank.result pdt-mwe-gold-standard.dat
#
# Pavel Pecina <pecina@ufal.mff.cuni.cz>, Feb 18, 2008
#
###############################################################################

use strict;

if ($#ARGV != 1) {
  die "Usage: eval.pl rank.list pdt-mwe-gold-standard.dat\n";
}

my $file1 = $ARGV[0];
my $file2 = $ARGV[1];

my ($lem1, $tag1, $rel1, $lem2, $tag2, $rel2, $A1, $A2, $A3, $A, $V);
my $key;
my %truth;
my @res;

my $N     = 0;    # size of the n-best list
my $POS   = 0;    # total positives

my $TP    = 0;    # true positives
my $FN    = 0;    # false negatives
my $FP    = 0;    # false positives

my $P     = 0;    # precision
my $APsum = 0;    # sum for average precision



if ($file1) {
  open( RANK, $file1 ) or die "Error openning the rank file: $file1 !\n";
} else { 
  die "Please specify the rank file!\n";
}


if ($file2) {
  open( TRUTH, $file2 ) or die "Error openning the gold standard file: $file2 !\n";
} else {
  die "Please specify the gold standard file!\n";
}

# load the gold standard data (indication of True Positives and True Negatives)

while (<TRUTH>) {

  # parse each input line
  ($lem1, $tag1, $rel1, $lem2, $tag2, $rel2, $A1, $A2, $A3, $A) = split; 

  # candidate identifier
  $key = join ("\t", ($lem1, $tag1, $rel1, $lem2, $tag2, $rel2));

  # store it
  $truth{$key} = $A;
}


# Read the ranked list of MWE candidates and store the indication of
# TP and TN (1 or 0)

while (<RANK>) {

  # parse the input line
  ($lem1, $tag1, $rel1, $lem2, $tag2, $rel2, $V) = split; 

  # candidate identifier
  $key = join ("\t", ($lem1, $tag1, $rel1, $lem2, $tag2, $rel2));

  # store the indication of TP and TN (1 or 0)
  $res[$N] = $truth{$key};

  # count Total Positives
  $POS++ if ( $truth{$key} > 0 );
  
  # count the size of the list
  $N++;
}

# Go through the list of 1s and 0s, update values of TP, FN, and FP,
# compute Precision after each True Positive candidate and add it to the
# partial sum needed to compute Average Precision

foreach (my $i; $i<$N; $i++) {

  if ($res[$i]) {
    
    $TP++;              # update number of True Positives
    $P = $TP/($TP+$FP); # compute Precision 
    $APsum += $P             # sum Precision for computing Average Precision later

  } else {
    
    $FP++;              # update number of False Positives

  }
}

# Print the result

printf "AP: %.2f\n", $APsum/$POS*100;

close(TRUTH);
close(RANK);

